/* =================================================================================
 *
 * USNAP:  Fast unique dense region detection and its application to lung cancer
 * Authors: Serene W. H. Wong, Chiara Pastrello, Max Kotlyar, Christos Faloutsos, Igor Jurisica
 *
 * =================================================================================
 */

import java.io.BufferedReader;
import java.io.FileReader;
import java.io.IOException;
import java.util.*;
import java.lang.Math;

public class dynamicGraphS2 {

    private static final int NOCOLLAPSE = 1;  //static variable is to the class, not the instance
    private static final double EPSILON = 0.000001;  //for float comparison

    private int V;  //no. of vertex for the collapsed graph
    private int E;  //no. of edge for the collapsed grpah
    private int Eu;  //no. of edge that is unique to Gu in the collapsed grpah
    private double overallWeight;  //weight of the collapsed grpah
    private int timestep; //no. of time steps in the dynamic graph
    private final float MAXWEIGHT;  //max weight, corr to the edge weight that is unique to Gu
    private HashMap<String, Integer> edgeH = new HashMap<String, Integer>();  //key is edge (first<second, concat 2 int), value is the number of graph that has this edge
    private HashMap<String, Integer> edgeUniqueH = new HashMap<String, Integer>(); //key is the edge (first<second, concat 2 int) in Gu, value is 1 which is a dummy
    private  HashMap<Integer, Triple> degreeH  = new HashMap<Integer, Triple>(); //key is the vertex, and Triple is dw, du, dv (in this order)
    private  HashMap<String, Integer> stringInt   = new HashMap<String, Integer>();  //map node from String to Int (input geneids treat as String)
    private  HashMap<Integer, String> intString  = new HashMap<Integer,String>();  //map node from Int back to String (input geneids treat as String)

    private ArrayList<HashMap<Integer, Map<Integer, Float>>> dG = new ArrayList<HashMap<Integer, Map<Integer, Float>>>();  //only dG 0 will be used, it is the collapsed graph
    
    /**  
     * creates graph from input file
     * @param files - input file contains path to input geneid files
     * The format for geneid files is geneid1\tgeneid2\tcorr\tabscorr but this will only consider the first 2 columns.  Error will throw if there is less than 2 cols.  The input file contains 1 line of header.  Geneids can be strings.
     * @param  timesteps - no. of timesteps
     * @param  conditionu - desired unique condition, wrt to the order in files.  e.g, conditionu=1 (numbering starts from 0) means that the 2nd file in files is the desired unique condition
     * Each graph in a condition/time point will be turned to a simple graph, ie loops and duplicate edges are ignored/skipped; vertices that are only with loops will not be included (e.g., in a single time point graph, 1000-1000 is the only edge that involves the node 1000, then 1000 will not be included in the vertices)
     * throws IOException
     */
    public dynamicGraphS2(final String files, final int timesteps, final int conditionu) throws IOException{
	long starttime = System.currentTimeMillis();

	this.V=0;
	this.E=0;
	this.Eu=0;
	this.overallWeight=0;
	this.timestep=timesteps;
	float maxweightTemp=(float)(2*(1+Math.log(this.timestep)));
	this.MAXWEIGHT=(Math.round(maxweightTemp * 100))/((float)100);
	int filecounter=0;  //which time graph
	

	final BufferedReader f = new BufferedReader(new FileReader(files));  //read in the files
	while(true){
            final String line = f.readLine();
	    if(line==null)
                break;
	    System.out.println("Reading in file: " + line);
	

	    final BufferedReader ff = new BufferedReader(new FileReader(line));  //read in geneid
	    HashMap<String, Integer> edgeSet = new HashMap<String, Integer>(); //check duplicate edges in a given condition, String is the edge(first<second, concat 2 int), Integer is a dummy
	    //read the header
	    final String header = ff.readLine();
	    

	    while(true){  
		
		final String line2 = ff.readLine();
		
		if(line2==null)
		    break;
				
		final String[] tokens = line2.split("\t");  //delim is tab
		if(tokens.length < 2) { //must have 2 cols for geneids
		    System.out.println("Skipped Line: " + line2);
		    continue;
		} 
	
		//only want geneid1,geneid2
		String geneid1=tokens[0];
		String geneid2=tokens[1];

		String e;
		if (geneid1.equals(geneid2))  //ignore loops, go to next edge  
		    continue;
		else if (geneid1.compareTo(geneid2) < 0 ){ //geneid1 before geneid2 lexi
		    e = geneid1+"-"+geneid2;
		}
		else {
		    e = geneid2+"-"+geneid1;
		}

		//not a duplicate edge in a given condition
		if (!edgeSet.containsKey(e)){  
		    edgeSet.put(e, 1);
				    
		    //for edge hash
		    if (edgeH.containsKey(e)){
			int value = edgeH.get(e);
			value=value+1;
			edgeH.put(e,value); 
		    }
		    else{
			edgeH.put(e,1);
		    }
		}

		//for Gu
		if (filecounter == conditionu){
		    if (!edgeUniqueH.containsKey(e)){  //duplicate edge will not be included
			edgeUniqueH.put(e,1); 
			
		    }
		}	
	    } //while geneid

	    edgeSet.clear();
	    filecounter++;
	}//while file names

	createCollapsedgraph();  //create collapsed graph
	initializeDegreeH();  //initialize degreeH
    
	long timetook = System.currentTimeMillis() - starttime;
	System.out.println("Input was loaded, collapsed graph was created: " + timetook + " milliseconds were taken.");
    }  //public dynamicGraph


    /**
     * create collapsed graph
     */
    public void createCollapsedgraph(){
	HashMap<Integer, Map<Integer, Float>> node = new HashMap<Integer, Map<Integer, Float>>();

	for (Map.Entry<String,Integer> e1 : edgeUniqueH.entrySet()){
	    String innerkey=e1.getKey();
	    int innervalue=e1.getValue();
	    int n1=-1;  //node in int
	    int n2=-1;  //node in int

	    String[] nodetoken = innerkey.split("-");
	    if (nodetoken.length < 2) //should not happen
		continue;

	    //put into hashes if it's not in already
	    if (!stringInt.containsKey(nodetoken[0])){
		stringInt.put(nodetoken[0], this.V);
		intString.put(this.V, nodetoken[0]);
		this.V++;
	    } 
	    if (!stringInt.containsKey(nodetoken[1])){
		stringInt.put(nodetoken[1], this.V);
		intString.put(this.V, nodetoken[1]);
		this.V++;
	    }

	    n1 = stringInt.get(nodetoken[0]);  //get node's int mapping
	    n2 = stringInt.get(nodetoken[1]);  //get node's int mapping

	    //update E
	    this.E++;

	    float weight = 0;
	    float weighttemp=0;
	    if (edgeH.containsKey(innerkey)){  //edgeH should contain innerkey
		if (edgeH.get(innerkey) == 1){ //edge ONLY present in Gu
		    //weight = (float)(2*(1+Math.log(this.timestep/innervalue)));
		    weight = this.MAXWEIGHT;
		    this.Eu++;
		}
		else if (edgeH.get(innerkey) > 1){
		    float tempvalue=(float)(edgeH.get(innerkey));
		    weighttemp =(float)(1+Math.log(this.timestep/tempvalue));
		    weight=(Math.round(weighttemp * 100))/((float)100);
		}
	    }
	    this.overallWeight=this.overallWeight+weight;
	    

	    //adjacency list-if the node doesn't exist, then add new node
	    Map<Integer, Float> edge = node.get(n1);
	    if(edge == null){
		edge = new HashMap<Integer, Float>();
		node.put(n1, edge);
	    }
	    edge.put(n2, weight);  //geneid2 is the key, value is the weight of the edge
	
	    //adjacency list-undirected
	    Map<Integer, Float> edge2 = node.get(n2);
	    if(edge2 == null){
		edge2 = new HashMap<Integer, Float>();
		node.put(n2, edge2);
	    }
	    edge2.put(n1, weight);

	}  //for edgeUniqueH
	dG.add(0,node);  //index 0 in dG is the collapsed graph
    }


    /**
     * for testing
     * printing out the stringInt hash
     */
    public void printstringInt(){
	for(Map.Entry<String, Integer> d:stringInt.entrySet()) {
	    String key = d.getKey();
	    int value = d.getValue();
	    System.out.println("stringInt:  key + value "+key+" "+value);
	}
    }

    /**
     * for testing
     * printing out the intString hash
     */
    public void printintString(){
	for(Map.Entry<Integer,String> d:intString.entrySet()) {
	    int key = d.getKey();
	    String value = d.getValue();
	    System.out.println("intString:  key + value "+key+" "+value);
	}
    }
    
    /**
     * for testing
     * printing out the dynamic Graph
     */
    public void printdynamicGraph(){
	for(int i = 0; i< NOCOLLAPSE; i++) {
	    System.out.println("adj list for NOCOLLAPSE: " + i);
	    for(Map.Entry<Integer, Map<Integer,Float>> n:dG.get(i).entrySet()) {
		int outerkey = n.getKey();
		for (Map.Entry<Integer,Float> e : n.getValue().entrySet()){
		    int innerkey=e.getKey();
		    float innervalue=e.getValue();
		    System.out.println("OuterKey: " + outerkey + " InnerKey: " + innerkey+ " VALUE:" +innervalue);
		}
	    }
	}
    }

    /**
     * my note:  to map the int back to String for final output
     * @return HasMap intString
     */
    public HashMap nodeIntToString(){
     return this.intString;
    }

     /**
     * initialize degreeH
     * only dG 0 is used, which is the collapsed graph
     */
    public void initializeDegreeH(){
	Triple <Float, Integer, Integer> currNodeInfo;  //dw, du, dv
	float dw=0;
	int du=0;
	int dv=0;

	for(Map.Entry<Integer, Map<Integer,Float>> n:dG.get(0).entrySet()) {
	    int outerkey = n.getKey();  //this is the vertex
	    dv=dG.get(0).get(outerkey).size();

	    for (Map.Entry<Integer,Float> e : n.getValue().entrySet()){
		int innerkey=e.getKey();
		float innervalue=e.getValue();

		//edge unique in Gu
		if ( (Math.abs(innervalue - this.MAXWEIGHT)) < EPSILON){ 
		    du++;
		}
		
		dw=dw + innervalue;	
	    }  //for innerkey
	    currNodeInfo=new Triple(dw, du, dv);
	    degreeH.put(outerkey, currNodeInfo);
	    
      	    //reset variable for next vertex in node
	    currNodeInfo=null;
	    dw=0;
	    du=0;
	    dv=0;
	}  //for outerkey
    }

    /**
     * for testing
     * printing out degreeH
     */
    public void printdegreeH(){
	System.out.println("in printdegreeH");
	//for(Map.Entry<Integer, Map<Integer,Float>> n:dG.get(i).entrySet()) {
	for(Map.Entry<Integer, Triple> d: degreeH.entrySet()) {
	    int key = d.getKey();
	    Triple value = d.getValue();
	    
	    float dw = (float)value.getFirst();
	    int du=(int)value.getSecond();
	    int dv=(int)value.getThird();
	    
	    System.out.println("Key: " + key + " dw, du, dv:" + dw + " " + du +" " + dv);
	}
    }

    /**
     * @param int t is the t_th time graph, (only t=0 is used for the collapsed graph)
     * @param int v is the vertex
     * @return all neighbors of v in time grpah t if v exists
     * @return null otherwise
     */
    public int[] neighbors(int t, int v){
	if (!dG.get(t).containsKey(v)){ //if time graph t doesn't contain v
	    return null;
	} else{
	    int NoNeighbors=dG.get(t).get(v).size();
	    int[] neighbor = new int[NoNeighbors];
	    int count=0;
	    for (int key : dG.get(t).get(v).keySet()) {
		neighbor[count]=key;
		count++;
	    }
	    return neighbor;
	}
    }
	
    /**
     * my notes:  for initializing the indexedBinaryHeap  
     * @param int t is the t_th time graph, (only t=0 is used for the collapsed graph)
     * @param int v is the vertex
     * @return the number of neighbors of v in time grpah t
     */
    public int NoOfneighbors(int t, int v){
	if (!dG.get(t).containsKey(v)){  //if time graph t doesn't contain v
	    return -1;
	}
	else{
	    int NoNeighbors=dG.get(t).get(v).size();
	    return NoNeighbors;
	}
    }

    /**
     * my notes:  for initializing the indexedBinaryHeap  
     * @param int v is the vertex
     * @return dw(v)*(1 + du(v)/d(v))
     */
    public float removalFunction(int v){
	if (!degreeH.containsKey(v)){  //if degreeH doesn't have vertex v
	    return -1;
	}
	else{
	    Triple degree = degreeH.get(v);
	    float dw = (float)degree.getFirst();
	    int du = (int)degree.getSecond();
	    int dv = (int)degree.getThird();
	    float ans = dw*(1+ (((float)du)/dv));
	    return ans;
	}
    }


    /**
     * remove all nodes in v from all time graphs (only t=0 is used)
     * @param int [] v vertices to be removed
     * return int vertices that are also removed becuase of the input.  If the adjacency list has 134->136,200 and 136->134, if I remove node 134, node 136 will also be removed.  This happens when 134's neighbors is invovled in a last edge.  Some nodes in vremoveAlso can overlap with int[] v
     */
    public int[] removeNodes(int[]v){
		
	HashSet<Integer> vremoveAlso = new HashSet<Integer>();
	for(int i = 0; i<NOCOLLAPSE; i++) {  //only i=0 will be used for the collapsed graph
	    for (int j=0; j<v.length; j++){  //remove the vertex
		if (dG.get(i).containsKey(v[j])){ //if time graph i contains v[j], remove it
		    
		    for (int key : dG.get(i).get(v[j]).keySet()) { //needed for undirected graph
			
			float tempweight = dG.get(i).get(key).get(v[j]);
			
			dG.get(i).get(key).remove(v[j]);

			//update E, Eu, key's degreeH
			this.E--; 
			Triple temp = degreeH.get(key);
			
			float dw = (float)temp.getFirst() - tempweight;
			int du = (int)temp.getSecond();
			int dv = (int)temp.getThird() -1;
			
			if ( (Math.abs(tempweight-this.MAXWEIGHT)) < EPSILON){
			    this.Eu--;
			    du=du-1;
			}

			this.overallWeight=this.overallWeight-tempweight;
			Triple temp2 =new Triple(dw, du, dv);
			degreeH.put(key, temp2);


			//delete another node although this doesn't change the number of edge deleted
			if(dG.get(i).get(key).size()==0){ 
			    dG.get(i).remove(key);
			    vremoveAlso.add(key);
			    this.V--;  
			    degreeH.remove(key);
			    
			}
		    }  //for keyset
		    dG.get(i).remove(v[j]);
		    this.V--;  
		    degreeH.remove(v[j]);
		} //if
		
	    } //for v
	} //for time
	

	int[] vremoveAlsoArray = new int[vremoveAlso.size()];
	int count=0;
	Iterator it = vremoveAlso.iterator();
	while (it.hasNext()){
	    vremoveAlsoArray[count]=(int)(it.next());
	    count++;
	}
	return vremoveAlsoArray;
    }

    /**
     * @param float threshold is the user input threshold for the exclusive condition
     * @return true if the exclusive condition is met, and false otherwise
     */
    public boolean exclusive(float threshold){
	if ( ( ((float)this.Eu)/this.E ) >= threshold){
	    return true;
	}
	else {
	    return false;
	}
    }

     /**
     * @return density
     */
    public float density(){
	float density;

	if (this.V != 0){
	    density=(float)((this.overallWeight)/(this.V));
	}
	else{
	    density=Integer.MIN_VALUE;
	}

	return density;
    }

    /**
     * @return E
     */ 
    public int getNoEdges(){
	return this.E;
    }

    /**
     * for graph time 0
     * @return V
     */
    public int getNoVertex(){
	return this.V;
    }


    /**
     * @return all vertices in dynamic graph, in time 0 (only time 0 is used in dG)
     * @return null otherwise
     */
    public int[] getAllVertices(){
	int t=0; //only time 0 is used in dG
	if (dG.get(t).isEmpty()){ //doesn't contain key
	    return null;
	} else{
	    int novertex=dG.get(t).size();
	    int[] vertex = new int[novertex];
	    int count=0;
	    for (int key : dG.get(t).keySet()) {
		vertex[count]=key;
		count++;
	    }
	    return vertex;
	}
    }

    //test copy graph
    public int getEu(){
	return this.Eu;
    }
    public float getoverallweight(){
	return ((float)(this.overallWeight));
    }
    public int gettimestep(){
	return this.timestep;
    }
    public float getmaxweight(){
	return this.MAXWEIGHT;
    }
 
    /**  
     * creates a copy of the original graph
     * not share the same reference, so dG from original change will not affect dG in the newly copied graph
     * @param  file to be copied
     */
    public dynamicGraphS2 (dynamicGraphS2 original){
	this.V = original.V;
	this.E = original.E;
	this.Eu = original.Eu;
	this.overallWeight = original.overallWeight;
	this.timestep= original.timestep;
	this.MAXWEIGHT = original.MAXWEIGHT;

	this.dG= new ArrayList<HashMap<Integer, Map<Integer, Float>>>(original.dG.size());
	for(int i = 0; i< original.NOCOLLAPSE; i++) {
	    
	    HashMap<Integer, Map<Integer, Float>> node = new HashMap<Integer, Map<Integer, Float>>();
	    for(Map.Entry<Integer, Map<Integer,Float>> n:original.dG.get(i).entrySet()) {
		for (Map.Entry<Integer,Float> e : n.getValue().entrySet()){
		    Map<Integer, Float> edge = node.get(n.getKey());
		    if(edge == null){
			edge = new HashMap<Integer, Float>();
			node.put(n.getKey(), edge);
		    }
		    edge.put(e.getKey(), e.getValue());  
		}
	    }
	    this.dG.add(i, node);
	}

	this.degreeH  = new HashMap<Integer, Triple>(original.degreeH.size()); 
	for(Map.Entry<Integer, Triple> d:original.degreeH.entrySet()) {
	    int key = d.getKey();
	    Triple value = d.getValue();
	    
	    this.degreeH.put(key, value);
	}

	this.stringInt = new HashMap<String, Integer>(original.stringInt.size());
	for(Map.Entry<String, Integer> d:original.stringInt.entrySet()) {
	    String key = d.getKey();
	    int value = d.getValue();
	    
	    this.stringInt.put(key, value);
	}

	this.intString = new HashMap<Integer,String>(original.intString.size());
	for(Map.Entry<Integer,String> d:original.intString.entrySet()) {
	    int key = d.getKey();
	    String value = d.getValue();
	    
	    this.intString.put(key, value);
	}

    }
    
    
}


