/* =================================================================================
 *
 * USNAP:  Fast unique dense region detection and its application to lung cancer
 * Authors: Serene W. H. Wong, Chiara Pastrello, Max Kotlyar, Christos Faloutsos, Igor Jurisica
 *
 * =================================================================================
 */

import java.io.BufferedWriter;
import java.io.File;
import java.io.FileWriter;
import java.io.IOException;
import java.util.*;


public class sdregion2 {

    public static void main(String[] args) throws IOException {
        if(args.length != 8) {
            System.err.println("Usage: input noOfSdregion noOfRestart PercentForOverlap conditionu exclusiveThreshold output timestep");
            System.exit(-1);
        }

        final String inputf = args[0];
        System.out.println("Input file: " + inputf);

	final int k = Integer.valueOf(args[1]);
        System.out.println("noOfSdregion: " + k);

	final int restart = Integer.valueOf(args[2]);
        System.out.println("noOfRestart: " + restart);

	final float overlapP = Float.valueOf(args[3]);
        System.out.println("PercentForOverlap: " + overlapP);

	final int conditionu = Integer.valueOf(args[4]);
        System.out.println("conditionu: " + conditionu);

	final float exclusiveThreshold = Float.valueOf(args[5]);
        System.out.println("exclusiveThreshold: " + exclusiveThreshold);

        final String outputf = args[6];
        System.out.println("Output file: " + outputf);

	final int timestep = Integer.valueOf(args[7]);
        System.out.println("TimeSteps: " + timestep);

	//read in input
	System.out.println("Reading the input dynamic graph...");
	dynamicGraphS2 G = new dynamicGraphS2(inputf,timestep,conditionu);
	
	new sdregion2().run(G, k, restart, overlapP, exclusiveThreshold, outputf);
    }


    public void run(dynamicGraphS2 G, final int k, final int restart, final float overlap, final float exclusiveThreshold, final String output) throws IOException {

	long starttime = System.currentTimeMillis();
	dynamicGraphS2 oriGraph = new dynamicGraphS2(G);  //copy graph
	final HashMap intToString=G.nodeIntToString();  //node's "int to String" mapping from dynamicGraphS

	int bcounter= 0;  //counter for no. of sregion
	int restartcounter=0;  //counter for no. of restarts
	ArrayList<Pair<int[],Float>> answers = new ArrayList<Pair<int[],Float>>();  //holds the final answer
	Set<Integer> allnodes = new HashSet<Integer>(); //all nodes in G
	HashMap<Integer, Integer> IntHeapid = new HashMap<Integer, Integer>();  //key is int (from dynamicGraphS2), value is heap id
	HashMap<Integer, Integer> HeapidInt = new HashMap<Integer, Integer>();  //key is heap id, value is int (the int from dynamicGraphS2)

	
	while ((bcounter < k) && (oriGraph.getNoVertex()> 1)) { 
	  
	    //get all nodes from oriGraph
	    int [] v = oriGraph.getAllVertices();
	    
	    //put all nodes of oriGraph 
	    //put in hash allnodes,for removing nodes to get the solution
	    //put in IntHeapid, HeapidInt
	    if (v != null){  //shouldn't be null as satisfies while loop, no. vertex > 1
		for(int i = 0; i< v.length; i++) {
		    allnodes.add(v[i]);
		    IntHeapid.put(v[i],i);
		    HeapidInt.put(i,v[i]);
		}
	    }

	    Pair <int[], Float> oneblock = findOneBlock(G, IntHeapid, HeapidInt, exclusiveThreshold);

	    //no s-block found, try to remove a node to continue
	    //if ((oneblock == null)||(oneblock.getSecond()==0)){
	    if (oneblock == null){
		if (oriGraph.getNoVertex() == 2){
		    break;  //algorithm finishes because even if I remove a node, the other node become isolated (with no edge) and will be removed as well.  my implmentation is for a simple graph,and disallow 2 isolated nodes, so if there are 2 nodes, it will be an edge in between them
		}
		else if (restartcounter < restart){
		    //int rnode=new Random().nextInt(v.length);
		    int rnode=0;  //remove the first node in v, v should have > 1 node as satisfies while loop.  want deterministic alg thus choose first node instead of a random node
		    int[] rnodeArray = {v[rnode]};  //put in array so can use the removeNodes function
		    int[] removeAlso = oriGraph.removeNodes(rnodeArray);
  		    restartcounter++;
		}
		else{  //restarted too many times already
		    break;  //algorithm finishes
		}
	    }
	    //entire adjdGorg is a s-region with max density returned from findOneBlock.  array length >=1 if oneblock is not null
	    else if ( ((oneblock.getFirst()).length == 1) && ((oneblock.getFirst())[0]==-1) ) {
		Pair <int[], Float> temp = new Pair(v, oneblock.getSecond());
		answers.add(bcounter,temp); 
		bcounter++;
		break;  //algorithm finishes
	    }
	    else{
		//remove oneblock nodes from allnodes
		int[] temp = oneblock.getFirst();
	        for (int i=0; i<temp.length; i++){
		    allnodes.remove(temp[i]);  //remove if present, thus the first -1 is fine because it won't do anything
		}
		//allnodes now contains the solution for one block
		int count=0;
		int [] tempnode2=new int [allnodes.size()];
		Iterator it = allnodes.iterator();
		while (it.hasNext()){
		    tempnode2[count]=(int)(it.next());
		    count++;
		}
		Pair <int[], Float> pairtemp = new Pair(tempnode2, oneblock.getSecond());
		answers.add(bcounter,pairtemp);
		bcounter++;
		
		//remove the solution from original graph
		int no = (int)(Math.ceil((1-overlap)*(tempnode2.length)));
		//int removenode[]=myRandom(tempnode2, no);
		int removenode[]=new int [no];
		for(int i=0; i<removenode.length; i++){  
		    removenode[i]=tempnode2[i];
		}
		int[] gone= oriGraph.removeNodes(removenode);
  
	    } //else
	    
	    //reset variables for next iteration
	    G = new dynamicGraphS2(oriGraph);  //copy graph
	   
	    
	    allnodes.clear();
	    IntHeapid.clear();
	    HeapidInt.clear();
	}  //while

	//the algorithm finds less sregions than what the user specified
	if (bcounter < k){
	    System.out.println("The algorithm is unable to find "+ k+" sdregion(s).  It is able to find "+bcounter+" sdregion(s).");
	}
		
	//long timetook=(System.currentTimeMillis() - starttime + 0.0)/1000;
	//System.out.println("Running time: " + timetook + "  seconds");
	long timetook=System.currentTimeMillis() - starttime;
	System.out.println("Running time: " + timetook + "  milliseconds");

	System.out.println("Writing outputs...");
	starttime = System.currentTimeMillis();
	//writeOutput(output, answers);
	writeOutput(output, answers, intToString);
	//timetook=(System.currentTimeMillis() - starttime + 0.0)/1000;
	timetook=System.currentTimeMillis() - starttime;
	System.out.println("Output was written: " + timetook+ " milliseconds were taken.");
    } //run
    
    
 

    /**
     * find one sregion from the given currG
     * @param currG
     * @param intHeapid, key is int (from dynamicGraphS2), value is heap id
     * @param heapidInt, key is heap id, value is int (the int from dynamicGraphS2)
     * @param exclusiveThreshold, user input for threshold for the exclusive condition
     * @return
     * @throws IOException
     */
    protected Pair findOneBlock(dynamicGraphS2 currG, HashMap intHeapid, HashMap heapidInt, float exclusiveThreshold) throws IOException {
	float densityMax=-1;  //the maximum density
	float densityCurr=-1;  //curent density
	int densityMaxIndex=-1;  //the count in which density was maximized
	Set<Integer> removedSet = new LinkedHashSet<Integer>(); //nodes remove, order of insert is kept
	int removedSetCount=0;  //how many items are in removedSet currently 
	Pair<Integer,Integer> pairRemove;
		
	IMinHeap heap = createHeap(currG, intHeapid);

	//to the start off currG
	if (currG.exclusive(exclusiveThreshold)){  //test the exclusive condition in the start off matrices
	    removedSet.add(-1);  //Nothing
	    removedSetCount=removedSetCount+1;
	    densityMax=currG.density();  //will always have at least 1 vertex
	    densityCurr=densityMax;
	    densityMaxIndex=1;
	}

	//remove node by node
	while ( currG.getNoVertex()> 2){ //loop until 2 nodes left, ie last edge
	    pairRemove= heap.poll(); //which node to remove, and remove from heap
	    int nodeRemove = pairRemove.getFirst();  //heap id 
	    int nodeRemoveInt = (int)(heapidInt.get(nodeRemove));
	    int[] nodeRemoveArray = {nodeRemoveInt};  //put in array so can use remove node function

	    //get neighbors before actual removal of the node, 0 is for dG 0 
	    int[] tempneighbor=currG.neighbors(0, nodeRemoveInt);  
  
	    //remove node from currG
	    int[] removeAlso=currG.removeNodes(nodeRemoveArray);  
	   	   
	   removedSet.add(nodeRemoveInt);  
	   removedSetCount++;

	   //remove nodes in removeAlso from heap as well
	   if (removeAlso != null){
	       for (int i=0;i<removeAlso.length; i++){
		   
		   removedSet.add(removeAlso[i]);  
		   heap.updatePriority( (int)(intHeapid.get(removeAlso[i])),Integer.MIN_VALUE);
		   
		   Pair<Integer,Integer> pairRemovetemp = heap.poll();
		   
		   
	       }
	       removedSetCount=removedSetCount+removeAlso.length;
	   }
	   
	   //update neighbors in the heap as well (after dynamicGraphS's removeNodes has updated these neighbors)
	   for(int j = 0; j <tempneighbor.length; j++) {
	       if(!removedSet.contains(tempneighbor[j])){  //if neighbor has not been removed yet
		   int neighborheapId = (int)(intHeapid.get(tempneighbor[j]));  //heap id for neighbor
		   float updatePriorityValue=currG.removalFunction(tempneighbor[j]);
		   int updatePriorityValueInt=(int)(Math.round(updatePriorityValue * 100.0));
		   heap.updatePriority(neighborheapId,updatePriorityValueInt);
	       }
	   }

	    //for testing heap
	    /*int [] vv = currG.getAllVertices();
	    for(int j = 0; j <vv.length; j++) {
		int priority = heap.getPriority( (int)(intHeapid.get(vv[j])) );
		System.out.println("in findoneblock check heap - vertex(int), heapid,priority : "+vv[j]+" "+intHeapid.get(vv[j])+" "+priority);
		}*/
	    //end testing heap


	    //compute block
	    densityCurr=currG.density();  //density for collapsed graph
	    if (densityMax < densityCurr){  //densityMax will never be less than -ve infinity, thus ok even if densityCurr is -ve infinity
		if (currG.exclusive(exclusiveThreshold)){
		    densityMax=densityCurr;
		    densityMaxIndex=removedSetCount;
		}
	    }
	   
	   
	}  //while > 2

	//no s-region found, ie densityMaxIndex was not assigned besides inititaion 
	if (densityMaxIndex == -1){ 
	    return null;
	} else{
	    int count=0;
	    Iterator it = removedSet.iterator();
	    int[] answer = new int [densityMaxIndex];
	    while (it.hasNext()){
		answer[count]=(int)(it.next());
		count++;
		if (count == densityMaxIndex){
		    break;
		}
	    }
	    //the algorithm won't find the same block more than once
	    Pair<int[],Float> pairanswer=new Pair(answer,densityMax); 
	    return pairanswer;
	}
   }
	
    /**
     * create heap
     * @param dynamic graph
     * @return
     * heap values are 100 times what dw*(1+ (du/dv)) are
     * dw*(1+ (du/dv)) is multiple by 100 then round to nearest long (we cast it to int)
     */
    private static IMinHeap createHeap(final dynamicGraphS2 g, final HashMap intHeapId) {
	IMinHeap heap = new HashIndexedMinHeap(g.getNoVertex());
	int[] vertex  = g.getAllVertices();  //store all vertices in array vertex
	float degreeFtn =0;
	int degreeFtnInt =0;  //change degreeFtn to int as heap takes in int
	//int intForGeneid=-1;

	for(int index = 0; index < vertex.length; index++) {
	    degreeFtn = g.removalFunction(vertex[index]);
	    degreeFtnInt = (int)(Math.round(degreeFtn * 100.0)); //Math.round returns the long, this is the same as round to 2 decimals and * 100 to make it whole number
	    int heapid = (int)(intHeapId.get(vertex[index]));
	    heap.insert(heapid, degreeFtnInt);
	}
        return heap;
    }
    
        

/**
     * write blocks found to the given output folder
     * @param output    output path
     * @param answer    ArrayList<Pair<int[],Float>> answers
     * @throws IOException
     */
    private static void writeOutput(String output, ArrayList<Pair<int[],Float>> answer, HashMap intTOstring) throws IOException {
	//write block

	BufferedWriter bw = new BufferedWriter(new FileWriter(output));
	

	//header
	bw.write("sdregionNo."+"\t"+"sdregionNodes"+"\t"+"density"+"\n");

	for(int i = 0; i < answer.size(); i++) { //loop through regions
	    int[] region = answer.get(i).getFirst();  //the nodes of the sdregion
	    bw.write("sdregion "+ i+"\t");
	    for(int j = 0; j < region.length; j++) {
		bw.write(intTOstring.get(region[j])+" ");
	    }
	    bw.write("\t");
	    Float density=answer.get(i).getSecond();
	    String densityString=density.toString();
	    bw.write(densityString);
	    bw.newLine();
        }

	bw.close();
    }
}
    
 
